/*
 * Decompiled with CFR 0.152.
 */
package net.tomp2p.relay;

import java.net.InetSocketAddress;
import java.util.Collection;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentHashMap;
import net.tomp2p.connection.ConnectionConfiguration;
import net.tomp2p.connection.Dispatcher;
import net.tomp2p.connection.PeerConnection;
import net.tomp2p.connection.Responder;
import net.tomp2p.connection.SignatureFactory;
import net.tomp2p.futures.FutureResponse;
import net.tomp2p.message.Buffer;
import net.tomp2p.message.Message;
import net.tomp2p.p2p.Peer;
import net.tomp2p.peers.Number160;
import net.tomp2p.peers.PeerAddress;
import net.tomp2p.peers.PeerSocketAddress;
import net.tomp2p.peers.PeerStatusListener;
import net.tomp2p.relay.BaseRelayForwarderRPC;
import net.tomp2p.relay.RconRPC;
import net.tomp2p.relay.RelayType;
import net.tomp2p.relay.RelayUtils;
import net.tomp2p.relay.android.AndroidForwarderRPC;
import net.tomp2p.relay.android.MessageBufferConfiguration;
import net.tomp2p.relay.android.gcm.IGCMSender;
import net.tomp2p.relay.android.gcm.RemoteGCMSender;
import net.tomp2p.relay.tcp.OpenTCPForwarderRPC;
import net.tomp2p.rpc.DispatchHandler;
import net.tomp2p.rpc.RPC;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RelayRPC
extends DispatchHandler {
    private static final Logger LOG = LoggerFactory.getLogger(RelayRPC.class);
    private final Peer peer;
    private final ConnectionConfiguration config;
    private final MessageBufferConfiguration bufferConfig;
    private final Map<Number160, BaseRelayForwarderRPC> forwarders;
    private final RconRPC rconRPC;
    private final IGCMSender gcmSenderRPC;

    public RelayRPC(Peer peer, RconRPC rconRPC, IGCMSender gcmSenderRPC, MessageBufferConfiguration bufferConfig, ConnectionConfiguration config) {
        super(peer.peerBean(), peer.connectionBean());
        this.peer = peer;
        this.gcmSenderRPC = gcmSenderRPC;
        this.bufferConfig = bufferConfig;
        this.config = config;
        this.forwarders = new ConcurrentHashMap<Number160, BaseRelayForwarderRPC>();
        this.rconRPC = rconRPC;
        this.register(new int[]{RPC.Commands.RELAY.getNr()});
    }

    /*
     * Enabled force condition propagation
     * Lifted jumps to return sites
     */
    public void handleResponse(Message message, PeerConnection peerConnection, boolean sign, Responder responder) throws Exception {
        LOG.debug("Received RPC message {}", (Object)message);
        if (message.type() == Message.Type.REQUEST_1 && message.command() == RPC.Commands.RELAY.getNr()) {
            if (message.intList().isEmpty()) {
                throw new IllegalArgumentException("Setup message should contain an integer value specifying the type");
            }
            Integer deviceType = message.intAt(0);
            if (deviceType.intValue() == RelayType.OPENTCP.ordinal()) {
                this.handleSetupTCP(message, peerConnection, responder);
                return;
            } else {
                if (deviceType.intValue() != RelayType.ANDROID.ordinal()) throw new IllegalArgumentException("Unknown relay type: " + deviceType);
                this.handleSetupAndroid(message, peerConnection, responder);
            }
            return;
        } else if (message.type() == Message.Type.REQUEST_2 && message.command() == RPC.Commands.RELAY.getNr()) {
            this.handlePiggyBackedMessage(message, responder);
            return;
        } else if (message.type() == Message.Type.REQUEST_3 && message.command() == RPC.Commands.RELAY.getNr()) {
            this.handleMap(message, responder);
            return;
        } else if (message.type() == Message.Type.REQUEST_4 && message.command() == RPC.Commands.RELAY.getNr()) {
            this.handleBufferRequest(message, responder);
            return;
        } else {
            if (message.type() != Message.Type.REQUEST_5 || message.command() != RPC.Commands.RELAY.getNr()) throw new IllegalArgumentException("Message content is wrong");
            this.handleLateResponse(message, peerConnection, sign, responder);
        }
    }

    public Peer peer() {
        return this.peer;
    }

    private SignatureFactory signatureFactory() {
        return this.connectionBean().channelServer().channelServerConfiguration().signatureFactory();
    }

    private Dispatcher dispatcher() {
        return this.peer().connectionBean().dispatcher();
    }

    public Set<PeerAddress> unreachablePeers() {
        HashSet<PeerAddress> unreachablePeers = new HashSet<PeerAddress>(this.forwarders.size());
        for (BaseRelayForwarderRPC forwarder : this.forwarders.values()) {
            unreachablePeers.add(forwarder.unreachablePeerAddress());
        }
        return unreachablePeers;
    }

    private void handleSetupTCP(Message message, PeerConnection peerConnection, Responder responder) {
        if (this.peerBean().serverPeerAddress().isRelayed()) {
            LOG.warn("I cannot be a relay since I'm relayed as well! {}", (Object)message);
            responder.response(this.createResponseMessage(message, Message.Type.DENIED));
            return;
        }
        OpenTCPForwarderRPC tcpForwarder = new OpenTCPForwarderRPC(peerConnection, this.peer, this.config);
        this.registerRelayForwarder(tcpForwarder);
        LOG.debug("I'll be your relay! {}", (Object)message);
        responder.response(this.createResponseMessage(message, Message.Type.OK));
    }

    private void handleSetupAndroid(Message message, PeerConnection peerConnection, Responder responder) {
        if (message.bufferList().size() < 1) {
            LOG.error("Device {} did not send any GCM registration id", (Object)peerConnection.remotePeer());
            responder.response(this.createResponseMessage(message, Message.Type.DENIED));
            return;
        }
        String registrationId = RelayUtils.decodeString(message.buffer(0));
        if (registrationId == null) {
            LOG.error("Cannot decode the registrationID from the message");
            responder.response(this.createResponseMessage(message, Message.Type.DENIED));
            return;
        }
        Integer mapUpdateInterval = message.intAt(1);
        if (mapUpdateInterval == null) {
            LOG.error("Android device did not send the peer map update interval.");
            responder.response(this.createResponseMessage(message, Message.Type.DENIED));
            return;
        }
        IGCMSender sender = null;
        if (message.neighborsSetList().isEmpty()) {
            if (this.gcmSenderRPC == null) {
                LOG.error("This relay is unable to serve unreachable Android devices because no GCM Authentication Key is configured");
                responder.response(this.createResponseMessage(message, Message.Type.DENIED));
                return;
            }
            sender = this.gcmSenderRPC;
        } else {
            Collection gcmServers = message.neighborsSet(0).neighbors();
            sender = new RemoteGCMSender(this.peer, this, this.config, gcmServers);
        }
        LOG.debug("Hello Android device! You'll be relayed over GCM. {}", (Object)message);
        AndroidForwarderRPC forwarderRPC = new AndroidForwarderRPC(this.peer, peerConnection.remotePeer(), this.bufferConfig, registrationId, sender, mapUpdateInterval);
        this.registerRelayForwarder(forwarderRPC);
        responder.response(this.createResponseMessage(message, Message.Type.OK));
    }

    private void registerRelayForwarder(BaseRelayForwarderRPC forwarder) {
        for (RPC.Commands command : RPC.Commands.values()) {
            if (command == RPC.Commands.RCON) {
                this.dispatcher().registerIoHandler(this.peer.peerID(), forwarder.unreachablePeerId(), (DispatchHandler)this.rconRPC, new int[]{command.getNr()});
                continue;
            }
            if (command == RPC.Commands.RELAY) {
                this.dispatcher().registerIoHandler(this.peer.peerID(), forwarder.unreachablePeerId(), (DispatchHandler)this, new int[]{command.getNr()});
                continue;
            }
            this.dispatcher().registerIoHandler(this.peer.peerID(), forwarder.unreachablePeerId(), (DispatchHandler)forwarder, new int[]{command.getNr()});
        }
        this.peer.peerBean().addPeerStatusListener((PeerStatusListener)forwarder);
        this.forwarders.put(forwarder.unreachablePeerId(), forwarder);
    }

    private void handlePiggyBackedMessage(Message message, final Responder responderToRelay) throws Exception {
        List peerSocketAddresses = message.peerSocketAddresses();
        InetSocketAddress sender = !peerSocketAddresses.isEmpty() ? PeerSocketAddress.createSocketTCP((PeerSocketAddress)((PeerSocketAddress)peerSocketAddresses.iterator().next())) : new InetSocketAddress(0);
        Buffer requestBuffer = message.buffer(0);
        Message realMessage = RelayUtils.decodeRelayedMessage(requestBuffer, message.recipientSocket(), sender, this.signatureFactory());
        realMessage.restoreContentReferences();
        LOG.debug("Received message from relay peer: {}", (Object)realMessage);
        final Message envelope = this.createResponseMessage(message, Message.Type.OK);
        Responder responder = new Responder(){

            public void response(Message responseMessage) {
                LOG.debug("Send reply message to relay peer: {}", (Object)responseMessage);
                try {
                    if (responseMessage.sender().isRelayed() && !responseMessage.sender().peerSocketAddresses().isEmpty()) {
                        responseMessage.peerSocketAddresses(responseMessage.sender().peerSocketAddresses());
                    }
                    envelope.buffer(RelayUtils.encodeMessage(responseMessage, RelayRPC.this.signatureFactory()));
                }
                catch (Exception e) {
                    LOG.error("Cannot piggyback the response", (Throwable)e);
                    this.failed(Message.Type.EXCEPTION, e.getMessage());
                }
                responderToRelay.response(envelope);
            }

            public void failed(Message.Type type, String reason) {
                responderToRelay.failed(type, reason);
            }

            public void responseFireAndForget() {
                responderToRelay.responseFireAndForget();
            }
        };
        DispatchHandler dispatchHandler = this.dispatcher().associatedHandler(realMessage);
        if (dispatchHandler == null) {
            responder.failed(Message.Type.EXCEPTION, "handler not found, probably not relaying peer anymore");
        } else {
            dispatchHandler.handleResponse(realMessage, null, false, responder);
        }
    }

    private void handleMap(Message message, Responder responder) {
        LOG.debug("Handle foreign map update {}", (Object)message);
        Collection map = message.neighborsSet(0).neighbors();
        BaseRelayForwarderRPC forwarder = this.forwarders.get(message.sender().peerId());
        if (forwarder == null) {
            LOG.error("No forwarder for peer {} found. Need to setup relay first");
            responder.response(this.createResponseMessage(message, Message.Type.NOT_FOUND));
            return;
        }
        forwarder.setPeerMap(RelayUtils.unflatten(map, message.sender()));
        if (message.neighborsSet(1) != null && forwarder instanceof AndroidForwarderRPC) {
            ((AndroidForwarderRPC)forwarder).changeGCMServers(message.neighborsSet(1).neighbors());
        }
        Message response = this.createResponseMessage(message, Message.Type.OK);
        responder.response(response);
    }

    private void handleBufferRequest(Message message, Responder responder) {
        LOG.debug("Handle buffer request of unreachable peer {}", (Object)message.sender());
        BaseRelayForwarderRPC forwarderRPC = this.forwarders.get(message.sender().peerId());
        if (forwarderRPC instanceof AndroidForwarderRPC) {
            AndroidForwarderRPC androidForwarder = (AndroidForwarderRPC)forwarderRPC;
            try {
                Message response = this.createResponseMessage(message, Message.Type.OK);
                response.buffer(androidForwarder.collectBufferedMessages());
                LOG.debug("Responding all buffered messages to Android device {}", (Object)message.sender());
                responder.response(response);
            }
            catch (Exception e) {
                LOG.error("Cannot respond with buffered messages.", (Throwable)e);
                responder.response(this.createResponseMessage(message, Message.Type.EXCEPTION));
            }
        } else {
            responder.failed(Message.Type.EXCEPTION, "This message type is intended for buffering forwarders only");
        }
    }

    private void handleLateResponse(Message message, PeerConnection peerConnection, boolean sign, Responder responder) {
        if (!message.sender().isSlow() || message.bufferList().isEmpty()) {
            throw new IllegalArgumentException("Late response does not come from slow peer or does not contain the buffered message");
        }
        Message realMessage = null;
        try {
            realMessage = RelayUtils.decodeRelayedMessage(message.buffer(0), message.recipientSocket(), message.senderSocket(), this.signatureFactory());
        }
        catch (Exception e) {
            LOG.error("Cannot decode the late response", (Throwable)e);
            responder.response(this.createResponseMessage(message, Message.Type.EXCEPTION));
            return;
        }
        LOG.debug("Received late response from slow peer: {}", (Object)realMessage);
        Map pendingRequests = this.dispatcher().getPendingRequests();
        FutureResponse pendingRequest = (FutureResponse)pendingRequests.remove(realMessage.messageId());
        if (pendingRequest != null) {
            pendingRequest.response(realMessage);
            LOG.debug("Successfully answered pending request {} with {}", (Object)pendingRequest.request(), (Object)realMessage);
            responder.response(RelayRPC.createResponseMessage((Message)message, (Message.Type)Message.Type.OK, (PeerAddress)message.recipient()));
        } else if (this.peer().peerAddress().isSlow()) {
            LOG.error("No pending request found for message {}. Ignore it.", (Object)realMessage);
        } else {
            BaseRelayForwarderRPC forwarder = this.forwarders.get(realMessage.recipient().peerId());
            if (forwarder == null) {
                LOG.error("Forwarder for the relayed peer not found. Cannot send late response {}", (Object)realMessage);
                responder.response(this.createResponseMessage(message, Message.Type.NOT_FOUND));
            } else {
                LOG.debug("We're just a relay peer. Send wrapped late response to requester wrapper: {} content: {}", (Object)message, (Object)realMessage);
                message.restoreBuffers();
                forwarder.forwardToUnreachable(message);
            }
        }
    }
}

