package net.tomp2p.relay;

import java.net.InetSocketAddress;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;

import net.tomp2p.connection.ChannelCreator;
import net.tomp2p.connection.ConnectionConfiguration;
import net.tomp2p.connection.DefaultConnectionConfiguration;
import net.tomp2p.connection.PeerConnection;
import net.tomp2p.connection.Responder;
import net.tomp2p.futures.BaseFutureAdapter;
import net.tomp2p.futures.FutureDone;
import net.tomp2p.futures.FuturePeerConnection;
import net.tomp2p.futures.FutureResponse;
import net.tomp2p.message.Buffer;
import net.tomp2p.message.Message;
import net.tomp2p.message.Message.Type;
import net.tomp2p.message.NeighborSet;
import net.tomp2p.p2p.Peer;
import net.tomp2p.peers.Number160;
import net.tomp2p.peers.PeerAddress;
import net.tomp2p.peers.PeerSocketAddress;
import net.tomp2p.peers.PeerStatatistic;
import net.tomp2p.rpc.DispatchHandler;
import net.tomp2p.rpc.RPC;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class RelayRPC extends DispatchHandler {

    private static final Logger LOG = LoggerFactory.getLogger(RelayRPC.class);
    private final ConnectionConfiguration config;
    private final Peer peer;

    /**
	 * This variable is needed, because a relay overwrites every RPC of an
	 * unreachable peer with another RPC called {@link RelayForwarderRPC}. This
	 * variable is forwarded to the {@link RelayForwarderRPC} in order to
	 * guarantee the existence of a {@link RconRPC}. Without this variable, no
	 * reverse connections would be possible.
	 * 
	 * @author jonaswagner
	 */
	private final RconRPC rconRPC;

	/**
     * Register the RelayRPC. After the setup, the peer is ready to act as a
     * relay if asked by an unreachable peer.
     * 
     * @param peer
     *            The peer to register the RelayRPC
     * @return
     */
	public RelayRPC(Peer peer, RconRPC rconRPC) {
        super(peer.peerBean(), peer.connectionBean());
        register(RPC.Commands.RELAY.getNr());
        this.peer = peer;
		this.rconRPC = rconRPC;
        config = new DefaultConnectionConfiguration();
    }

    /**
     * Send the peer map of an unreachable peer to a relay peer, so that the
     * relay peer can reply to neighbor requests on behalf of the unreachable
     * peer.
     * 
     * @param peerAddress
     *            The peer address of the relay peer
     * @param map
     *            The unreachable peer's peer map.
     * @param fcc
     * @return
     */
    public FutureResponse sendPeerMap(PeerAddress peerAddress, List<Map<Number160, PeerStatatistic>> map, final PeerConnection peerConnection) {
        final Message message = createMessage(peerAddress, RPC.Commands.RELAY.getNr(), Type.REQUEST_3);
        message.keepAlive(true);
        // TODO: neighbor size limit is 256, we might have more here
        message.neighborsSet(new NeighborSet(-1, RelayUtils.flatten(map)));
        final FutureResponse futureResponse = new FutureResponse(message);
		return RelayUtils.sendSingle(peerConnection, futureResponse, peerBean(), connectionBean(), config);
    }

    /**
     * Forward a message through the open peer connection to the unreachable
     * peer.
     * 
     * @param peerConnection
     *            The open connection to the unreachable peer
     * @param buf
     *            Buffer of the message that needs to be forwarded to the
     *            unreachable peer
     * @return
     */
    public FutureResponse forwardMessage(final PeerConnection peerConnection, final Buffer buf, final PeerSocketAddress peerSocketAddress) {
        final Message message = createMessage(peerConnection.remotePeer(), RPC.Commands.RELAY.getNr(), Type.REQUEST_2);
        message.keepAlive(true);
        message.buffer(buf);
        Collection<PeerSocketAddress> peerSocketAddresses = new ArrayList<PeerSocketAddress>(1);
        //this will be read RelayRPC.handlePiggyBackMessage
        peerSocketAddresses.add(peerSocketAddress);
        message.peerSocketAddresses(peerSocketAddresses);
        final FutureResponse futureResponse = new FutureResponse(message);
        return RelayUtils.sendSingle(peerConnection, futureResponse, peerBean(), connectionBean(), config);
    }

    /**
     * Set up a relay connection to a peer. If the peer that is asked to act as
     * relay is relayed itself, the request will be denied.
     * 
     * @param channelCreator
     * @param fpcshall
     *            FuturePeerConnection to the peer that shall act as a relay.
     * @return FutureDone with a peer connection to the newly set up relay peer
     */
    public FutureDone<PeerConnection> setupRelay(final ChannelCreator channelCreator, FuturePeerConnection fpc) {
        final FutureDone<PeerConnection> futureDone = new FutureDone<PeerConnection>();
        final Message message = createMessage(fpc.remotePeer(), RPC.Commands.RELAY.getNr(), Type.REQUEST_1);
        message.keepAlive(true);
        final FutureResponse futureResponse = new FutureResponse(message);
        LOG.debug("Setting up relay connection to peer {}, message {}", fpc.remotePeer(), message);

        fpc.addListener(new BaseFutureAdapter<FuturePeerConnection>() {
            public void operationComplete(final FuturePeerConnection futurePeerConnection) throws Exception {
                if (futurePeerConnection.isSuccess()) {
                	final PeerConnection peerConnection = futurePeerConnection.object();
					RelayUtils.sendSingle(peerConnection, futureResponse, peerBean(), connectionBean(), config).addListener(
							new BaseFutureAdapter<FutureResponse>() {
                        public void operationComplete(FutureResponse future) throws Exception {
                            if (future.isSuccess()) {
                                futureDone.done(peerConnection);
                            } else {
                                futureDone.failed(future);
                            }
                        }
                    });
                } else {
                    futureDone.failed(futurePeerConnection);
                }
            }
        });
        return futureDone;
    }

    @Override
    public void handleResponse(final Message message, PeerConnection peerConnection, final boolean sign, Responder responder) throws Exception {
        LOG.debug("received RPC message {}", message);
        if (message.type() == Type.REQUEST_1 && message.command() == RPC.Commands.RELAY.getNr()) {
            handleSetup(message, peerConnection, responder);
        } else if (message.type() == Type.REQUEST_2 && message.command() == RPC.Commands.RELAY.getNr()) {
            handlePiggyBackMessage(message, responder);
        } else if (message.type() == Type.REQUEST_3 && message.command() == RPC.Commands.RELAY.getNr()) {
            handleMap(message, responder);
        } else {
            throw new IllegalArgumentException("Message content is wrong");
        }
    }

    public Peer peer() {
        return this.peer;
    }

    private void handleSetup(Message message, final PeerConnection peerConnection, Responder responder) {
        
        if (peerBean().serverPeerAddress().isRelayed()) {
            // peer is behind a NAT as well -> deny request
        	LOG.warn("I cannot be a relay since I'm relayed as well! {}", message);
            responder.response(createResponseMessage(message, Type.DENIED));
            return;
        }

        // register relay forwarder
        RelayForwarderRPC.register(peerConnection, peer, this, rconRPC);

        LOG.debug("I'll be your relay! {}", message);
        responder.response(createResponseMessage(message, Type.OK));
    }

    private void handlePiggyBackMessage(Message message, final Responder responderToRelay) throws Exception {
        // TODO: check if we have right setup
        Buffer requestBuffer = message.buffer(0);
        //this contains the real sender
        Collection<PeerSocketAddress> peerSocketAddresses = message.peerSocketAddresses();
        final InetSocketAddress sender;
        if(!peerSocketAddresses.isEmpty()) {
			sender = PeerSocketAddress.createSocketTCP(peerSocketAddresses.iterator().next());
        } else {
        	sender = new InetSocketAddress(0);
        }
        Message realMessage = RelayUtils.decodeMessage(requestBuffer, message.recipientSocket(), sender, connectionBean().channelServer().channelServerConfiguration().signatureFactory());
        
        //we don't call the decoder where the relay address is handled, so we need to do this on our own.
        boolean isRelay = realMessage.sender().isRelayed();
        if(isRelay && !realMessage.peerSocketAddresses().isEmpty()) {
        	PeerAddress tmpSender = realMessage.sender().changePeerSocketAddresses(realMessage.peerSocketAddresses());
        	realMessage.sender(tmpSender);
        }
        
        LOG.debug("Received message from relay peer: {}", realMessage);
        realMessage.restoreContentReferences();
        
        final Message response = createResponseMessage(message, Type.OK);
        final Responder responder = new Responder() {
        	
        	//TODO: add reply leak handler
        	@Override
        	public void response(Message responseMessage) {
        		LOG.debug("Send reply message to relay peer: {}", responseMessage);
        		try {
        			if(responseMessage.sender().isRelayed() && !responseMessage.sender().peerSocketAddresses().isEmpty()) {
        				responseMessage.peerSocketAddresses(responseMessage.sender().peerSocketAddresses());
        			}
	                response.buffer(RelayUtils.encodeMessage(responseMessage, connectionBean().channelServer().channelServerConfiguration().signatureFactory()));
                } catch (Exception e) {
                	failed(Type.EXCEPTION, e.getMessage());
	                e.printStackTrace();
                }
                responderToRelay.response(response);
        	}

			@Override
            public void failed(Type type, String reason) {
				responderToRelay.failed(type, reason);
	            
            }

			@Override
            public void responseFireAndForget() {
				responderToRelay.responseFireAndForget();
            }
        };
        // TODO: Not sure what to do with the peer connection and sign
        DispatchHandler dispatchHandler = peer.connectionBean().dispatcher().associatedHandler(realMessage);
        //boolean isRelay = realMessage.sender().isRelayed();
        /*if(isRelay && !realMessage.peerSocketAddresses().isEmpty()) {
        	PeerAddress tmpSender = realMessage.sender().changePeerSocketAddresses(realMessage.peerSocketAddresses());
        	realMessage.sender(tmpSender);
        }*/
        if(dispatchHandler == null) {
        	responder.failed(Type.EXCEPTION, "handler not found, probably not relaying peer anymore");
        } else {
        	dispatchHandler.handleResponse(realMessage, null, false, responder);
        }
    }

    /**
     * Updates the peer map of an unreachable peer on the relay peer, so that
     * the relay peer can respond to neighbor RPC on behalf of the unreachable
     * peer
     * 
     * @param message
     * @param responder
     */
    private void handleMap(Message message, Responder responder) {
    	LOG.debug("handle foreign map {}", message);
        Collection<PeerAddress> map = message.neighborsSet(0).neighbors();
        RelayForwarderRPC relayForwarderRPC = RelayForwarderRPC.find(peer, message.sender().peerId());
        if (relayForwarderRPC != null) {
            relayForwarderRPC.setMap(RelayUtils.unflatten(map, message.sender()));
        } else {
            LOG.error("need to call setup relay first");
        }
        Message response = createResponseMessage(message, Type.OK);
        responder.response(response);
    }
}
