/*
 * Copyright 2009 Thomas Bocek
 *
 * Licensed under the Apache License, Version 2.0 (the "License"); you may not
 * use this file except in compliance with the License. You may obtain a copy of
 * the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
 * License for the specific language governing permissions and limitations under
 * the License.
 */
package net.tomp2p.futures;


/**
 * Wraps a future into an other future. This is useful for futures that are
 * created later on. You can create a wrapper, return it to the user, create an
 * other future, wrap this created future and the wrapper will tell the user if
 * the newly created future has finished.
 * 
 * @author Thomas Bocek
 * @param <K>
 */
public class FutureWrapper2<K extends BaseFuture, J extends BaseFuture> extends BaseFutureImpl<K> {
    
    final private J wrappedFuture;
    
    public FutureWrapper2(J wrappedFuture) {
    	this.wrappedFuture = wrappedFuture;
    }
    
    public FutureWrapper2(K baseFuture, J wrappedFuture) {
    	this.wrappedFuture = wrappedFuture;
    	self(baseFuture);
    }

    /**
     * Wait for the future, which will cause this future to complete if the
     * wrapped future completes.
     * 
     * @param future
     *            The future to wrap
     */
    public void waitFor() {
    	if(wrappedFuture == null) {
    		return;
    	}
    	wrappedFuture.addListener(new BaseFutureAdapter<K>() {
            @Override
            public void operationComplete(final K future) throws Exception {
                synchronized (lock) {
                    if (!completedAndNotify()) {
                        return;
                    }
                    type = future.type();
                    reason = future.toString();
                }
                notifyListeners();
            }
        });
    }

    /**
     * @return The wrapped (original) future.
     */
    public J wrappedFuture() {
        return wrappedFuture;
    }
}
